
{*******************************************************}
{                                                       }
{                Delphi TLED Component                  }
{           Copyright  1997, Gary Hardman.             }
{                    2nd May 1997                       }
{                                                       }
{   No warranties whatsoever are implied or offered     }
{                    for this unit                      }
{                                                       }
{*******************************************************}

unit Led;

interface

uses

  WinTypes, WinProcs, Classes, Graphics, Controls, Messages;

type

  TLEDColor = (lcRed,lcGreen,lcYellow);   { LED colour is Red, Green or Yellow }

  TLED = class(TGraphicControl)
    private
  { Private declarations }

      fLit         : Boolean;             { LED On or Off }
      fColor       : TLEDColor;           { LED colour }
      fChange      : TNotifyEvent;        { Custom Event }

      procedure SetBounds(Left, Top, Width, Height: integer); override;
      procedure SetState(const NewState: Boolean); virtual;
      procedure SetColor(const NewColor: TLEDColor); virtual;
      procedure PaintLED(const IndexX: Integer; const IndexY : Integer);
      procedure CM_ENABLEDCHANGED(var Msg: TMessage); message cm_EnabledChanged;
      procedure CM_PARENTCOLORCHANGED(var Msg: TMessage); message cm_ParentColorChanged;

    protected
  { Protected declarations }
      procedure Paint; override;

    public
  { Public declarations }
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      function ChangeState : Boolean; virtual;
      function ChangeColor: TLEDColor; virtual;

    published
  { Published declarations }

      property Lit: Boolean                { Indicates Leds On or Off }
               read fLit
               write SetState
               default False;

      property Color: TLEDColor            { Indicates LED Green or Red }
               read fColor
               write SetColor
               default lcRed;

      property Enabled;
      property Visible;
      property Hint;
      property ParentShowHint;
      property ShowHint;
      property Tag;

      property OnClick;
      property OnDblClick;

      property OnChangeState: TNotifyEvent { User-defined Method }
               read fChange
               write fChange;

      property OnDragDrop;
      property OnDragOver;
      property OnEndDrag;
      property OnMouseDown;
      property OnMouseMove;
      property OnMouseUp;
  end;

implementation

{$IFDEF WIN32}
{$R LED_32.RES}
{$ELSE}
{$R LED_16.RES}
{$ENDIF}

type
  TLEDBitmap = class
  public
    LED_Pics: TBitmap;
    LED_List: TList;
    Trans_Color : TColor;
    constructor Create;
    destructor Destroy; override;
end;

const
  LEDBitmap : TLEDBitmap = nil;
  NumRows = 4;
  bmpHeight = 13;
  bmpWidth = 12;

constructor TLEDBitmap.Create;
begin
  inherited Create;
  LED_Pics := TBitmap.Create;
  LED_Pics.Handle:= LoadBitmap(hInstance, 'LED_PICS');
  Trans_Color := LED_PICS.TransparentColor;
  LED_List := TList.Create;
end;

destructor TLEDBitmap.Destroy;
begin
  LED_Pics.Free;
  LED_List.Free;
  inherited Destroy;
end;

constructor TLED.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
{ if the LED_PICS Bitmap has not already been created, do so now }
  if LEDBitmap = nil then LEDBitmap := TLEDBitmap.Create;
  LEDBitmap.LED_LIST.Add(Self); { Add this new LED instance to the list }
  fLit := False;
  FColor := lcRed;
  ControlStyle := ControlStyle + [csOpaque]      + [csDoubleClicks]
                               + [csFixedHeight] + [csFixedWidth];
end;

destructor TLED.Destroy;
begin
{ Delete the current control from the LED_LIST }
  with LEDBitmap.LED_LIST do
    Delete (IndexOf(Self));
  { If there are no more LEDs, delete the BITMAP object }
    if LEDBitmap.LED_LIST.Count = 0 then
    begin
      LEDBitmap.Free; { Free the Bitmap object }
      LEDBitmap:=nil; { and point to NIL }
    end;
 inherited Destroy;
end;

procedure TLED.SetBounds(Left,Top,Width,Height : Integer);
begin
{ Size of individual LED resource bitmaps }
  inherited SetBounds(Left,Top,bmpWidth,bmpHeight);
end;

procedure TLED.CM_ENABLEDCHANGED(var Msg: TMessage);
begin
  if Assigned(fChange) then OnChangeState(Self);
  inherited;
end;

procedure TLED.CM_PARENTCOLORCHANGED(var Msg: TMessage);
begin
  Canvas.Brush.Color := TWincontrol(Parent).Brush.Color;
  inherited;
end;

{ Set fLit state }
procedure TLED.SetState(const NewState : Boolean);
begin
 if fLit <> NewState then begin
   fLit:= NewState;
   if csDesigning in ComponentState then Invalidate else
   begin
     Paint;
     if Assigned(fChange) then OnChangeState(Self);
   end; { if not designing }
 end; { if fLit <> NewState }
end;

{ Set fColor }
procedure TLED.SetColor(const NewColor : TLEDColor);
begin
 if fColor <> NewColor then
 begin
   fColor:= NewColor;
   if csDesigning in ComponentState then Invalidate else
   begin
     Paint;
     if Assigned(fChange) then OnChangeState(Self);
   end; {if not designing}
 end; {if fColor <> NewColor}
end;

{ Toggle fLit state }
function TLED.ChangeState : Boolean;
begin
 fLit := not fLit;
 Paint;
 if Assigned(fChange) then OnChangeState(Self);
 Result := fLit;
end;

{ Cycle fColor through Red...Green...Yellow...Red... etc. }
function TLED.ChangeColor : TLEDColor;
begin
 case fColor of
   lcRed: fColor := lcGreen;
   lcGreen: fColor := lcYellow;
   lcYellow: fColor := lcRed;
 end; { case }
 Paint;
 if Assigned(fChange) then OnChangeState(Self); { Call User Method }
 Result := fColor;
end;

procedure TLED.Paint;
var
  IndexX, IndexY: Integer; { Identifies LED position in LED_PIC grid }
begin
  if not Enabled then PaintLed(0,NumRows-1) else { Disabled picture }
  begin
    IndexX := Ord(Lit);       { 0= Left, 1 = Right column in Bitmap }
    IndexY := Ord(fColor);    { 0= Red, 1 = Green, 2 = Yellow };
    PaintLed(IndexX, IndexY); { Paint the LED Bitmap 'Column, Row' section }
  end;

end;

procedure TLED.PaintLED(const IndexX: Integer; const IndexY: Integer);
begin
  Canvas.BrushCopy(ClientRect, LEDBitmap.LED_Pics,
                   Rect(bmpWidth*IndexX, bmpHeight*IndexY,
                        bmpWidth*(IndexX+1), bmpHeight*(IndexY+1)),
                   LEDBitmap.Trans_Color);
end;

end.
